using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.InputSystem;
using UnityEngine.InputSystem.EnhancedTouch;
using UnityEngine.UI;
using Touch = UnityEngine.InputSystem.EnhancedTouch.Touch;



public class GerenciadorDeEntrada : MonoBehaviour
{
    public RectTransform joystickArea; // Área do joystick
    public InputActionReference lookAction; // Ação de rotação
    public InputActionReference moveAction; // Ação de movimento (LeftStick)

    private Finger joystickFinger = null; // Armazena o dedo que controla o joystick
    private Finger lookFinger = null; // Armazena o dedo que controla a rotação
    private bool usingJoystick = false; // Flag para saber se o joystick está ativo

    private void Awake()
    {
        EnhancedTouchSupport.Enable();
    }

    private void OnEnable()
    {
        TouchSimulation.Enable();
        Touch.onFingerDown += HandleFingerDown;
        Touch.onFingerUp += HandleFingerUp;
        
        moveAction.action.performed += ctx => OnJoystickMove(ctx);
        moveAction.action.canceled += ctx => OnJoystickStop(ctx);
    }

    private void OnDisable()
    {
        TouchSimulation.Disable();
        Touch.onFingerDown -= HandleFingerDown;
        Touch.onFingerUp -= HandleFingerUp;

        moveAction.action.performed -= OnJoystickMove;
        moveAction.action.canceled -= OnJoystickStop;
    }

    private void HandleFingerDown(Finger finger)
    {
        Vector2 touchPosition = finger.screenPosition;

        // Se o toque for no joystick e ainda não temos um dedo associado a ele
        if (IsTouchInsideJoystick(touchPosition) && joystickFinger == null)
        {
            joystickFinger = finger; // Esse dedo agora controla o joystick

            if(lookFinger ==null)
                lookAction.action.ApplyBindingOverride(0, "<Touchscreen>/touch1/delta");  // Rotação usa Touch #1

        }
        // Se for um segundo toque e não estiver no joystick, ele controla a rotação
        else if (lookFinger == null)
        {
            lookFinger = finger; // Esse dedo agora controla a rotação
        }
    }


    private void HandleFingerUp(Finger finger)
    {
        if (finger == joystickFinger)
        {
            joystickFinger = null; // Liberamos o joystick para outro toque
            lookAction.action.ApplyBindingOverride(0, "<Touchscreen>/touch0/delta");  // Rotação usa Touch #1

        }
        else if (finger == lookFinger)
        {
            lookFinger = null; // Liberamos a rotação para outro toque
        }
    }

    private bool IsTouchInsideJoystick(Vector2 touchPosition)
    {
        RectTransformUtility.ScreenPointToLocalPointInRectangle(
            joystickArea, touchPosition, null, out Vector2 localPoint);

        return joystickArea.rect.Contains(localPoint);
    }

    private void OnJoystickMove(InputAction.CallbackContext context)
    {
        usingJoystick = true;
    }

    private void OnJoystickStop(InputAction.CallbackContext context)
    {
        usingJoystick = false;  
    }

}
